/**
 * InesDrive Servis – site.js
 * AJAX forms, scroll effects, mobile menu, reveal animations
 */

(function () {
    'use strict';

    // ─── SCROLL HEADER ─────────────────────────────────────
    const header = document.getElementById('header');
    if (header) {
        window.addEventListener('scroll', () => {
            header.classList.toggle('scrolled', window.scrollY > 50);
        });
    }

    // ─── HAMBURGER MENU ────────────────────────────────────
    const hamburger = document.getElementById('hamburger');
    const mainNav = document.getElementById('main-nav');
    if (hamburger && mainNav) {
        hamburger.addEventListener('click', () => {
            hamburger.classList.toggle('active');
            mainNav.classList.toggle('open');
        });
        // Close menu on link click
        mainNav.querySelectorAll('a').forEach(a => {
            a.addEventListener('click', () => {
                hamburger.classList.remove('active');
                mainNav.classList.remove('open');
            });
        });
    }

    // ─── SMOOTH SCROLL ─────────────────────────────────────
    document.querySelectorAll('a[href^="#"]').forEach(link => {
        link.addEventListener('click', (e) => {
            const target = document.querySelector(link.getAttribute('href'));
            if (target) {
                e.preventDefault();
                target.scrollIntoView({ behavior: 'smooth' });
            }
        });
    });

    // ─── REVEAL ON SCROLL ──────────────────────────────────
    const revealElements = document.querySelectorAll('.reveal');
    const revealObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('visible');
                revealObserver.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1, rootMargin: '0px 0px -50px 0px' });

    revealElements.forEach(el => revealObserver.observe(el));


    // ─── AJAX FORM SUBMISSION ──────────────────────────────

    function showFlash(containerId, message, type) {
        const container = document.getElementById(containerId);
        if (!container) return;
        container.innerHTML = `<div class="flash-msg ${type}">${message}</div>`;
        container.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }

    function setupAjaxForm(formId, flashId) {
        const form = document.getElementById(formId);
        if (!form) return;

        form.addEventListener('submit', async (e) => {
            e.preventDefault();

            const submitBtn = form.querySelector('[type="submit"]');
            const originalText = submitBtn ? submitBtn.value : '';
            if (submitBtn) {
                submitBtn.value = 'Odesílám…';
                submitBtn.disabled = true;
            }

            try {
                const data = new FormData(form);
                data.append('_xhr', '1');
                const action = form.getAttribute('action') || window.location.href;
                const response = await fetch(action, {
                    method: 'POST',
                    body: data,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                    },
                });

                const contentType = response.headers.get('content-type') || '';
                if (!contentType.includes('application/json')) {
                    // Non-JSON = form succeeded but returned HTML (redirect)
                    showFlash(flashId, 'Odesláno! Děkujeme.', 'success');
                    form.reset();
                    if (submitBtn) { submitBtn.value = originalText; submitBtn.disabled = false; }
                    return;
                }

                const result = await response.json();

                if (result.success) {
                    showFlash(flashId, result.message, result.type || 'success');
                    form.reset();
                    // Hide honeypot again
                    const hp = form.querySelector('[data-hp]');
                    if (hp) hp.value = '';
                } else {
                    showFlash(flashId, result.message || 'Nastala chyba.', 'error');
                }
            } catch (err) {
                showFlash(flashId, 'Nastala chyba při odesílání. Zkuste to prosím znovu.', 'error');
            }

            if (submitBtn) {
                submitBtn.value = originalText;
                submitBtn.disabled = false;
            }
        });
    }

    // Initialize AJAX forms
    setupAjaxForm('bookingForm', 'booking-flash');
    setupAjaxForm('contactForm', 'contact-flash');
    setupAjaxForm('subscribeForm', 'newsletter-flash');


    // ─── HONEYPOT HIDE ─────────────────────────────────────
    document.querySelectorAll('[data-hp]').forEach(el => {
        el.closest('.hp-field')?.style && (el.closest('.hp-field').style.display = 'none');
    });

    // ─── SERVICE CARD → BOOKING PRE-SELECT ──────────────────
    document.querySelectorAll('.service-card-btn[data-service]').forEach(btn => {
        btn.addEventListener('click', function () {
            const serviceId = this.getAttribute('data-service');
            const select = document.getElementById('booking-service');
            if (select && serviceId) {
                select.value = serviceId;
            }
        });
    });

})();
