<?php

declare(strict_types=1);

namespace App\UI\Homepage;

use App\Model;
use Nette;
use Nette\Application\UI\Form;

/**
 * Frontend homepage presenter – single-page layout with all sections.
 * Instalatérské potřeby - Petr Diviš
 */
class HomepagePresenter extends Nette\Application\UI\Presenter
{
	private string $siteEmail;

	public function __construct(
		private Model\ServiceRepository $services,
		private Model\ServiceCategoryRepository $categories,
		private Model\BookingRepository $bookings,
		private Model\MessageRepository $messages,
		private Model\EmailRepository $emails,
		private Model\AuditLogRepository $auditLog,
		private Nette\Mail\Mailer $mailer,
		private Model\Translator $translator,
	) {
	}

	public function injectSiteConfig(Nette\DI\Container $container): void
	{
		$this->siteEmail = $container->getParameters()['site']['email'];
	}


	/**
	 * Process forms via direct POST before render.
	 */
	public function actionDefault(?string $locale = null): void
	{
		if ($locale && in_array($locale, Model\Translator::LOCALES, true)) {
			$this->translator->setLocale($locale);
		}

		$post = $this->getHttpRequest()->getPost();

		if (!empty($post['_form'])) {
			$formName = $post['_form'];
			$isXhr = !empty($post['_xhr']);

			// Honeypot check
			if (!empty($post['website'])) {
				if ($isXhr) {
					$this->sendJson(['success' => true, 'message' => 'Děkujeme!', 'type' => 'success']);
				}
				$this->redirect('this');
			}

			switch ($formName) {
				case 'booking':
					$this->processBooking($post, $isXhr);
					break;
				case 'contact':
					$this->processContact($post, $isXhr);
					break;
				case 'subscribe':
					$this->processSubscribe($post, $isXhr);
					break;
			}
		}
	}


	public function renderDefault(): void
	{
		$this->template->serviceGroups = $this->services->findGroupedByCategory();
		$this->template->siteName = 'Instalatérské potřeby - Petr Diviš';
		$this->template->sitePhone = '+420 777 200 916';
		$this->template->siteAddress = 'Služeb 1, 100 00 Praha 10';
		$this->template->serviceSelect = $this->services->findForSelect();
		$this->template->t = $this->translator;
		$this->template->locale = $this->translator->getLocale();
		$this->template->locales = Model\Translator::LOCALES;
	}


	// ─── BOOKING (POPTÁVKA) ─────────────────────────────────

	private function processBooking(array $post, bool $isXhr): void
	{
		$name = trim($post['name'] ?? '');
		$phone = trim($post['phone'] ?? '');
		$email = trim($post['email'] ?? '');
		$serviceId = (int) ($post['service_id'] ?? 0);
		$note = trim($post['note'] ?? '');

		// Validation
		if ($name === '' || $phone === '') {
			if ($isXhr) {
				$this->sendJson(['success' => false, 'message' => 'Vyplňte prosím jméno a telefon.', 'type' => 'error']);
			}
			$this->flashMessage('Vyplňte prosím jméno a telefon.', 'error');
			$this->redirect('this#poptavka');
		}

		// Resolve service/category name
		$serviceText = '';
		if ($serviceId) {
			$svc = $this->services->findById($serviceId);
			$serviceText = $svc ? $svc->name : '';
		}

		$this->bookings->insert([
			'service_id' => $serviceId ?: null,
			'service_text' => $serviceText,
			'preferred_date' => null,
			'preferred_time' => null,
			'name' => $name,
			'email' => $email ?: null,
			'phone' => $phone,
			'car_info' => null,
			'note' => $note ?: null,
		]);

		// Send email
		$mailStatus = $this->sendNotification(
			'Nová poptávka – ' . $serviceText . ' – ' . $name,
			"Nová poptávka z webu petrdivis.cz\n\n"
			. "Kategorie: {$serviceText}\n"
			. "Jméno: {$name}\n"
			. "Telefon: {$phone}\n"
			. "E-mail: " . ($email ?: 'neuvedeno') . "\n\n"
			. "Popis poptávky:\n" . ($note ?: '—')
		);

		$this->auditLog->log('booking', $this->getHttpRequest()->getRemoteAddress(), $this->getHttpRequest()->getHeader('User-Agent'), $phone, $serviceText);

		$message = 'Děkujeme za vaši poptávku! Ozveme se vám co nejdříve.';
		if ($mailStatus !== 'ok') {
			$message .= ' (Email: ' . $mailStatus . ')';
		}

		if ($isXhr) {
			$this->sendJson(['success' => true, 'message' => $message, 'type' => $mailStatus === 'ok' ? 'success' : 'info']);
		}

		$this->flashMessage($message, 'success');
		$this->redirect('this#poptavka');
	}


	// ─── CONTACT ─────────────────────────────────────────────

	private function processContact(array $post, bool $isXhr): void
	{
		$subjects = [
			'general' => 'Obecný dotaz',
			'sortiment' => 'Dotaz na sortiment',
			'order' => 'Objednávka',
			'availability' => 'Dostupnost zboží',
			'other' => 'Jiné',
		];

		$name = trim($post['name'] ?? '');
		$email = trim($post['email'] ?? '');
		$phone = trim($post['phone'] ?? '');
		$subject = trim($post['subject'] ?? '');
		$messageText = trim($post['message'] ?? '');
		$subjectLabel = $subjects[$subject] ?? $subject;

		if ($name === '' || $email === '' || $phone === '' || $messageText === '') {
			if ($isXhr) {
				$this->sendJson(['success' => false, 'message' => 'Vyplňte prosím všechna povinná pole.', 'type' => 'error']);
			}
			$this->flashMessage('Vyplňte prosím všechna povinná pole.', 'error');
			$this->redirect('this#contact');
		}

		$this->messages->insert([
			'subject' => $subjectLabel,
			'name' => $name,
			'email' => $email,
			'phone' => $phone,
			'message' => $messageText,
		]);

		$mailStatus = $this->sendNotification(
			'Zpráva z webu – ' . $subjectLabel . ' – ' . $name,
			"Předmět: {$subjectLabel}\n"
			. "Jméno: {$name}\n"
			. "E-mail: {$email}\n"
			. "Telefon: {$phone}\n\n"
			. $messageText,
			$email,
			$name
		);

		$message = 'Děkujeme za vaši zprávu! Ozveme se vám co nejdříve.';
		if ($mailStatus !== 'ok') {
			$message .= ' (Email: ' . $mailStatus . ')';
		}

		if ($isXhr) {
			$this->sendJson(['success' => true, 'message' => $message, 'type' => 'success']);
		}

		$this->flashMessage($message, 'success');
		$this->redirect('this#contact');
	}


	// ─── NEWSLETTER ──────────────────────────────────────────

	private function processSubscribe(array $post, bool $isXhr): void
	{
		$realname = trim($post['realname'] ?? '');
		$email = trim($post['email'] ?? '');

		if ($realname === '' || $email === '') {
			if ($isXhr) {
				$this->sendJson(['success' => false, 'message' => 'Vyplňte prosím jméno a e-mail.', 'type' => 'error']);
			}
			$this->redirect('this#newsletter');
		}

		$existing = $this->emails->findByEmail($email);
		$message = '';
		$type = 'success';

		if ($existing) {
			if ($existing->active && $existing->verified) {
				$message = 'Tento e-mail je již přihlášen k odběru.';
				$type = 'info';
			} elseif (!$existing->verified) {
				$token = $existing->token ?: bin2hex(random_bytes(16));
				$existing->update(['token' => $token, 'realname' => $realname]);
				$this->sendVerificationEmail($email, $realname, $token);
				$message = 'Ověřovací e-mail byl znovu odeslán. Zkontrolujte svou schránku.';
			} else {
				$existing->update(['active' => 1]);
				$message = 'Odběr byl znovu aktivován. Děkujeme!';
			}
		} else {
			$token = bin2hex(random_bytes(16));
			$this->emails->insert([
				'realname' => $realname,
				'email' => $email,
				'active' => 0,
				'verified' => 0,
				'token' => $token,
				'created_at' => date('Y-m-d H:i:s'),
			]);
			$this->sendVerificationEmail($email, $realname, $token);
			$message = 'Děkujeme! Na váš e-mail jsme poslali ověřovací odkaz.';
		}

		$this->auditLog->log('newsletter', $this->getHttpRequest()->getRemoteAddress(), $this->getHttpRequest()->getHeader('User-Agent'), $email, $realname);

		if ($isXhr) {
			$this->sendJson(['success' => true, 'message' => $message, 'type' => $type]);
		}

		$this->flashMessage($message, $type);
		$this->redirect('this#newsletter');
	}


	// ─── MAIL HELPERS ────────────────────────────────────────

	private function sendNotification(string $subject, string $body, ?string $replyTo = null, ?string $replyName = null): string
	{
		$mail = new \Nette\Mail\Message;
		$mail->setFrom($this->siteEmail, 'Petr Diviš Web')
			->addTo($this->siteEmail)
			->setSubject($subject)
			->setBody($body);

		if ($replyTo) {
			$mail->addReplyTo($replyTo, $replyName ?? '');
		}

		try {
			$this->mailer->send($mail);
			return 'ok';
		} catch (\Exception $e) {
			return $e->getMessage();
		}
	}


	private function sendVerificationEmail(string $email, string $name, string $token): void
	{
		$link = $this->link('//Homepage:default', ['do' => 'verifyEmail', 'token' => $token]);

		$mail = new \Nette\Mail\Message;
		$mail->setFrom($this->siteEmail, 'Instalatérské potřeby - Petr Diviš')
			->addTo($email, $name)
			->setSubject('Potvrzení odběru – Petr Diviš')
			->setBody(
				"Dobrý den, {$name},\n\n"
				. "děkujeme za registraci k odběru novinek.\n"
				. "Pro potvrzení klikněte na následující odkaz:\n\n"
				. "{$link}\n\n"
				. "S pozdravem,\nInstalatérské potřeby - Petr Diviš"
			);

		try {
			$this->mailer->send($mail);
		} catch (\Exception $e) {
		}
	}


	public function handleVerifyEmail(string $token): void
	{
		$subscriber = $this->emails->findByToken($token);
		if ($subscriber) {
			$subscriber->update(['verified' => 1, 'active' => 1]);
			$this->flashMessage('E-mail byl úspěšně ověřen. Děkujeme!', 'success');
		} else {
			$this->flashMessage('Neplatný ověřovací odkaz.', 'error');
		}
		$this->redirect('Homepage:default');
	}


	public function actionUnsubscribe(string $token): void
	{
		$subscriber = $this->emails->findByToken($token);
		if ($subscriber) {
			$subscriber->update(['active' => 0]);
			$this->flashMessage('Byli jste úspěšně odhlášeni z odběru.', 'success');
		} else {
			$this->flashMessage('Neplatný odkaz pro odhlášení.', 'error');
		}
		$this->redirect('Homepage:default');
	}
}
