<?php

declare(strict_types=1);

namespace App\UI\Admin;

use App\Model;
use Nette;
use Nette\Application\UI\Form;

/**
 * Admin panel – manage services, bookings, messages, emails.
 */
class AdminPresenter extends Nette\Application\UI\Presenter
{
	public function __construct(
		private Model\ServiceRepository $services,
		private Model\ServiceCategoryRepository $categories,
		private Model\BookingRepository $bookings,
		private Model\MessageRepository $messages,
		private Model\EmailRepository $emails,
	) {
	}

	protected function startup(): void
	{
		parent::startup();
		if (!$this->getUser()->isLoggedIn()) {
			$this->redirect('Sign:in');
		}
	}


	// ─── DASHBOARD ───────────────────────────────────────────

	public function renderDefault(): void
	{
		$this->template->newBookings = $this->bookings->countNew();
		$this->template->unreadMessages = $this->messages->countUnread();
		$this->template->recentBookings = $this->bookings->findAll()->limit(5);
		$this->template->recentMessages = $this->messages->findAll()->limit(5);
	}


	// ─── SERVICES ────────────────────────────────────────────

	public function renderServices(): void
	{
		$this->template->categories = $this->categories->findAll()->fetchAll();
		$this->template->services = $this->services->findAll()->fetchAll();
	}

	public function renderEditService(int $id = 0): void
	{
		$this->template->serviceId = $id;
		$this->template->categories = $this->categories->findAll()->fetchPairs('id', 'name');

		if ($id) {
			$service = $this->services->findById($id);
			if (!$service) {
				$this->error('Služba nenalezena.');
			}
			$this['serviceForm']->setDefaults($service->toArray());
			$this->template->service = $service;
		}
	}

	protected function createComponentServiceForm(): Form
	{
		$form = new Form;

		$cats = $this->categories->findAll()->fetchPairs('id', 'name');
		$form->addSelect('category_id', 'Kategorie', $cats)
			->setRequired('Vyberte kategorii.');

		$form->addText('name', 'Název služby')
			->setRequired('Zadejte název.');

		$form->addText('name_uk', 'Název 🇺🇦 UK');
		$form->addText('name_en', 'Název 🇬🇧 EN');
		$form->addText('name_fr', 'Název 🇫🇷 FR');
		$form->addText('name_ru', 'Název 🇷🇺 RU');
		$form->addText('name_pl', 'Název 🇵🇱 PL');
		$form->addText('name_sk', 'Název 🇸🇰 SK');

		$form->addText('description', 'Popis');

		$form->addText('price_from', 'Cena od');

		$form->addInteger('weight', 'Pořadí')
			->setDefaultValue(0);

		$form->addCheckbox('active', 'Aktivní')
			->setDefaultValue(true);

		$form->addSubmit('save', 'Uložit');

		$form->onSuccess[] = [$this, 'serviceFormSucceeded'];

		return $form;
	}

	public function serviceFormSucceeded(Form $form, \stdClass $values): void
	{
		$id = (int) $this->getParameter('id');
		$data = (array) $values;

		if ($id) {
			$this->services->update($id, $data);
			$this->flashMessage('Služba byla aktualizována.', 'success');
		} else {
			$this->services->insert($data);
			$this->flashMessage('Služba byla přidána.', 'success');
		}

		$this->redirect('Admin:services');
	}

	public function handleDeleteService(int $id): void
	{
		$this->services->delete($id);
		$this->flashMessage('Služba byla smazána.', 'success');
		$this->redirect('Admin:services');
	}


	// ─── CATEGORIES ──────────────────────────────────────────

	public function renderEditCategory(int $id = 0): void
	{
		$this->template->categoryId = $id;
		if ($id) {
			$cat = $this->categories->findById($id);
			if (!$cat) {
				$this->error('Kategorie nenalezena.');
			}
			$this['categoryForm']->setDefaults($cat->toArray());
			$this->template->category = $cat;
		}
	}

	protected function createComponentCategoryForm(): Form
	{
		$form = new Form;
		$form->addText('name', 'Název kategorie')->setRequired('Zadejte název.');
		$form->addText('name_uk', 'Název 🇺🇦 UK');
		$form->addText('name_en', 'Název 🇬🇧 EN');
		$form->addText('name_fr', 'Název 🇫🇷 FR');
		$form->addText('name_ru', 'Název 🇷🇺 RU');
		$form->addText('name_pl', 'Název 🇵🇱 PL');
		$form->addText('name_sk', 'Název 🇸🇰 SK');
		$form->addText('icon', 'Ikona (emoji)')->setDefaultValue('🔧');
		$form->addInteger('weight', 'Pořadí')->setDefaultValue(0);
		$form->addCheckbox('active', 'Aktivní')->setDefaultValue(true);
		$form->addSubmit('save', 'Uložit');
		$form->onSuccess[] = [$this, 'categoryFormSucceeded'];
		return $form;
	}

	public function categoryFormSucceeded(Form $form, \stdClass $values): void
	{
		$id = (int) $this->getParameter('id');
		$data = (array) $values;

		if ($id) {
			$this->categories->update($id, $data);
			$this->flashMessage('Kategorie byla aktualizována.', 'success');
		} else {
			$this->categories->insert($data);
			$this->flashMessage('Kategorie byla přidána.', 'success');
		}

		$this->redirect('Admin:services');
	}

	public function handleDeleteCategory(int $id): void
	{
		$this->categories->delete($id);
		$this->flashMessage('Kategorie byla smazána.', 'success');
		$this->redirect('Admin:services');
	}


	// ─── BOOKINGS ────────────────────────────────────────────

	public function renderBookings(): void
	{
		$this->template->bookings = $this->bookings->findAll()->fetchAll();
	}

	public function handleBookingStatus(int $id, string $status): void
	{
		$this->bookings->updateStatus($id, $status);
		$this->flashMessage('Stav poptávky byl aktualizován.', 'success');
		$this->redirect('Admin:bookings');
	}

	public function handleSaveServiceTranslation(): void
	{
		$post = $this->getHttpRequest()->getPost();
		$id = (int) ($post['id'] ?? 0);
		$field = $post['field'] ?? '';
		$value = trim($post['value'] ?? '');

		if ($id && in_array($field, ['name', 'description', 'price_from', 'name_uk', 'name_en', 'name_fr', 'name_ru', 'name_pl', 'name_sk'], true)) {
			$this->services->update($id, [$field => $value]);
			$this->sendJson(['success' => true]);
		}
		$this->sendJson(['success' => false]);
	}


	// ─── MESSAGES ────────────────────────────────────────────

	public function renderMessages(): void
	{
		$this->template->messages = $this->messages->findAll()->fetchAll();
	}

	public function handleMarkRead(int $id): void
	{
		$this->messages->markAsRead($id);
		if ($this->isAjax()) {
			$this->sendJson(['success' => true]);
		}
		$this->redirect('Admin:messages');
	}


	// ─── EMAILS ──────────────────────────────────────────────

	public function renderEmails(): void
	{
		$this->template->emails = $this->emails->findAll()->order('created_at DESC')->fetchAll();
	}


	// ─── TRANSLATIONS ────────────────────────────────────────

	public function renderTranslations(string $locale = 'cs'): void
	{
		$locales = ['cs', 'uk', 'en', 'fr', 'ru', 'pl', 'sk'];
		$localeLabels = ['cs' => '🇨🇿 Čeština', 'uk' => '🇺🇦 Ukrajinština', 'en' => '🇬🇧 Angličtina', 'fr' => '🇫🇷 Francouzština', 'ru' => '🇷🇺 Ruština', 'pl' => '🇵🇱 Polština', 'sk' => '🇸🇰 Slovenština'];

		if (!in_array($locale, $locales, true)) {
			$locale = 'cs';
		}

		$file = __DIR__ . '/../../lang/' . $locale . '.neon';
		$translations = [];
		if (file_exists($file)) {
			$data = \Nette\Neon\Neon::decodeFile($file);
			$translations = $this->flattenArray($data);
		}

		$this->template->locale = $locale;
		$this->template->locales = $locales;
		$this->template->localeLabels = $localeLabels;
		$this->template->translations = $translations;
	}


	public function handleSaveTranslations(): void
	{
		$post = $this->getHttpRequest()->getPost();
		$locale = $post['locale'] ?? 'cs';
		$keys = $post['keys'] ?? [];
		$values = $post['values'] ?? [];

		if (!in_array($locale, ['cs', 'uk', 'en', 'fr', 'ru', 'pl', 'sk'], true)) {
			$this->flashMessage('Neplatný jazyk.', 'error');
			$this->redirect('Admin:translations');
		}

		// Rebuild nested array from flat keys
		$data = [];
		foreach ($keys as $i => $key) {
			$value = $values[$i] ?? '';
			$parts = explode('.', $key);
			$ref = &$data;
			foreach ($parts as $part) {
				if (!isset($ref[$part])) {
					$ref[$part] = [];
				}
				$ref = &$ref[$part];
			}
			$ref = $value;
			unset($ref);
		}

		$file = __DIR__ . '/../../lang/' . $locale . '.neon';
		$neon = "# " . strtoupper($locale) . " translations\n" . \Nette\Neon\Neon::encode($data, true);
		file_put_contents($file, $neon);

		// Clear template cache
		$cacheDir = __DIR__ . '/../../../temp/cache';
		if (is_dir($cacheDir)) {
			foreach (new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($cacheDir, \FilesystemIterator::SKIP_DOTS), \RecursiveIteratorIterator::CHILD_FIRST) as $item) {
				$item->isDir() ? @rmdir($item->getPathname()) : @unlink($item->getPathname());
			}
		}

		$this->flashMessage('Překlady uloženy.', 'success');
		$this->redirect('Admin:translations', ['locale' => $locale]);
	}


	private function flattenArray(array $array, string $prefix = ''): array
	{
		$result = [];
		foreach ($array as $key => $value) {
			$newKey = $prefix ? $prefix . '.' . $key : $key;
			if (is_array($value)) {
				$result = array_merge($result, $this->flattenArray($value, $newKey));
			} else {
				$result[$newKey] = $value;
			}
		}
		return $result;
	}
}
