<?php

declare(strict_types=1);

namespace App\Model;

use Nette;

/**
 * Simple NEON-based translator for multi-language support.
 */
class Translator implements Nette\Localization\Translator
{
	private string $locale = 'cs';
	private array $translations = [];
	private string $langDir;

	/** Supported locales */
	public const LOCALES = ['cs', 'en', 'de', 'pl', 'sl', 'ua'];
	public const DEFAULT_LOCALE = 'cs';

	public function __construct(string $appDir)
	{
		$this->langDir = $appDir . '/lang';
	}


	public function setLocale(string $locale): void
	{
		if (in_array($locale, self::LOCALES, true)) {
			$this->locale = $locale;
			$this->translations = [];
		}
	}


	public function getLocale(): string
	{
		return $this->locale;
	}


	/**
	 * Translate a key like "hero.title" or "nav.services".
	 */
	public function translate(\Stringable|string $message, mixed ...$parameters): string
	{
		$message = (string) $message;
		$translations = $this->getTranslations();

		// Support dot notation: "hero.title" -> $translations['hero']['title']
		$keys = explode('.', $message);
		$value = $translations;
		foreach ($keys as $key) {
			if (is_array($value) && array_key_exists($key, $value)) {
				$value = $value[$key];
			} else {
				return $message; // Key not found, return as-is
			}
		}

		if (!is_string($value)) {
			return $message;
		}

		// Simple sprintf replacement if parameters provided
		if ($parameters) {
			return sprintf($value, ...$parameters);
		}

		return $value;
	}


	private function getTranslations(): array
	{
		if (!$this->translations) {
			$file = $this->langDir . '/' . $this->locale . '.neon';
			if (file_exists($file)) {
				$this->translations = Nette\Neon\Neon::decodeFile($file);
			}
		}
		return $this->translations;
	}
}
